---
name: i18n-integration
description: Add internationalization (i18n) support to web applications with Chinese/English language switching. Use when User needs to implement i18n for Vue3, React applications, including: (1) Setting up i18n framework and dependencies, (2) Creating translation files and message resources, (3) Implementing language switching functionality, (4) Extracting and organizing translatable text, (5) Managing translation keys and namespaces
---

# i18n - Web Application Internationalization

## Overview

Add comprehensive internationalization support to web applications with focus on Chinese/English language switching. Supports Vue3, React, and Angular frameworks with framework-specific best practices.

## Framework Detection

Identify the framework first by examining:
- `package.json` dependencies
- Project structure and file extensions
- Configuration files

## Workflow Decision Tree

```
START
  │
  ├─ No i18n setup?
  │   └─ Go to "Initial Setup" for framework
  │
  ├─ Adding new translations?
  │   └─ Go to "Adding Translations"
  │
  ├─ Implementing language switcher?
  │   └─ Go to "Language Switcher Component"
  │
  └─ Extracting existing text?
      └─ Go to "Text Extraction"
```

## Initial Setup by Framework

### Vue3

**Dependencies:** `vue-i18n@latest`

**Setup steps:**
1. Install: `npm install vue-i18n@latest`
2. Create i18n instance (see `assets/vue3/i18n.js`)
3. Register in `main.js` or `main.ts`
4. Create locale files (see `assets/vue3/locales/`)
5. Use `$t()` in templates, `t()` in setup

**Reference:** See `references/vue3.md` for complete patterns

### React

**Dependencies:** `react-i18next` + `i18next`

**Setup steps:**
1. Install: `npm install i18next react-i18next`
2. Configure i18next (see `assets/react/i18n.js`)
3. Wrap app with provider (see `assets/react/I18nextProvider.jsx`)
4. Create translation files (see `assets/react/locales/`)
5. Use `useTranslation()` hook or `<Trans>` component

**Reference:** See `references/react.md` for complete patterns

### Naming Conventions

- Use **lowercase_with_underscores** for keys
- Group by **feature** or **page**
- Use **namespaces** for large apps (React/Vue3)
- Prefix **shared** keys with `common.`

### Interpolation

**Vue3:** `{{variable}}`
```vue
{{ $t('welcome', { name: userName }) }}
```

**React:** `{{variable}}`
```jsx
t('welcome', { name: userName })
```

## Language Switcher Component

Create a switcher with:

### Vue3 Example
```vue
<template>
  <select v-model="$i18n.locale" @change="saveLocale">
    <option value="en">English</option>
    <option value="zh">中文</option>
  </select>
</template>

<script setup>
import { useI18n } from 'vue-i18n'

const { locale } = useI18n()

const saveLocale = () => {
  localStorage.setItem('locale', locale.value)
}
</script>
```

### React Example
```jsx
import { useTranslation } from 'react-i18next'

function LanguageSwitcher() {
  const { i18n } = useTranslation()

  const changeLanguage = (lng) => {
    i18n.changeLanguage(lng)
    localStorage.setItem('locale', lng)
  }

  return (
    <select onChange={(e) => changeLanguage(e.target.value)}>
      <option value="en">English</option>
      <option value="zh">中文</option>
    </select>
  )
}
```

## Text Extraction

### Extract from Existing Code

1. Identify hardcoded strings in templates/components
2. Create meaningful translation keys
3. Add to appropriate locale files
4. Replace with i18n function calls

### Batch Extraction

For large codebases, use framework-specific extraction tools:

**Vue3:** Manual extraction or use `vue-i18n-extract` package
**React:** `i18next-parser` for automatic key extraction

## Plurals and Dates

### Pluralization

**Vue3:**
```json
{
  "apple": "no apples | one apple | {count} apples"
}
```
```vue
{{ $tn('apple', count, count) }}
```

**React:**
```json
{
  "apple_one": "one apple",
  "apple_other": "{{count}} apples"
}
```
```jsx
t('apple', { count })
```

### Date/Number Formatting

**Vue3:** Use `$d()` and `$n()`
```vue
{{ $d(new Date(), 'short') }}
{{ $n(price, 'currency') }}
```

**React:** Use `i18next` formatting
```jsx
const { t } = useTranslation()
// Format with libraries like date-fns, Intl API
```

## Best Practices

### DO

- Use **descriptive keys** (e.g., `user.profile.title` not `user.title1`)
- **Group translations** by feature/module
- Use **interpolation** for dynamic values
- Store **user's locale preference** in localStorage
- **Lazy load** locale files for large apps
- Use **namespaces** for code organization (React/Vue3)

### DON'T

- Don't use **numeric keys** (e.g., `message.001`)
- Don't **concatenate** translated strings
- Don't **hardcode** translations in components
- Don't **duplicate** common translations (use `common.*` namespace)
- Don't **mix languages** in same file
- Don't **nest keys too deep** (max 3 levels recommended)

## Resources

### Framework-Specific References

- **Vue3:** See `references/vue3.md` for vue-i18n patterns, composition API usage, lazy loading
- **React:** See `references/react.md` for react-i18next patterns, hook usage, context setup

### Templates

- **Vue3:** `assets/vue3/` - i18n instance, locale files, component examples
- **React:** `assets/react/` - i18n config, provider setup, translation files
